<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;

class GuarantorPostToEbanqr extends Command
{
    protected $signature = 'GuarantorPostToEbanqr:cron';

    protected $description = 'Post guarantor to Ebanqr';

    protected $http_headers;

    private $url;

    public function __construct()
    {
        parent::__construct();

        $this->http_headers = [
            'Authorization' => env('EB_AUTHORIZATION'),
            'Accept' => env('EB_ACCEPT'),
            'api_key' => env('EB_API_KEY'),
            'eBanqr-Platform-TenantID' => env('EB_TENANTID'),
        ];
        $this->url = env('EB_URL');
    }

    public function handle()
    {
        echo "{$this->signature} is running \n";

        $guarantor = DB::table('loan_guarantors')
            ->where('sync', '=', 0)
            ->get();

        foreach ($guarantor as $g)
        {
            $loan = DB::table('loans')
                ->where('id', '=', $g->loan_id)
                ->get()
                ->first();
            $to_post = [];
            if(is_null($g->client_id))
            {
                $to_post = [
                    "locale" => "en",
                    "dateFormat" => "dd MMMM yyyy",
                    "firstname" => $g->first_name,
                    "lastname" => $g->last_name,
                    "addressLine1" => $g->address,
                    "city" => $g->city,
                    "mobileNumber" => $g->phone_number,
                    "dob" => Carbon::parse($g->date_of_birth)->format('d M Y'),
                    "guarantorTypeId" => $g->guarantor_type_id,
                    "clientRelationshipTypeId" => $g->client_relationship_type_id
                ];
                echo "Syncing guarantor {$g->first_name} {$g->last_name}";
                $response = Http::withHeaders($this->http_headers)
                    ->withBody(json_encode($to_post), 'application/json')
                    ->post("{$this->url}/loans/{$loan->loan_id}/guarantors");

                if($response->ok())
                {
                    $api_response = json_decode($response, true);
                    $affected = DB::table('loan_guarantors')
                        ->where('id', $g->id)
                        ->update(['remote_id' => $api_response['resourceId'], 'sync' => 1]);
                    if($affected == 1)
                    {
                        echo "The guarantor of the loan application was updated in the database";
                    } else {
                        echo "The guarantor of the loan application was created in Ebanqr but not updated in the database";
                    }
                }
            } elseif (!is_null($g->client_id))
            {

                $to_post = [
                    "entityId" => $g->client_id,
                    "guarantorTypeId" => $g->guarantor_type_id,
                    "clientRelationshipTypeId" => $g->client_relationship_type_id
                ];

                echo "Syncing guarantor {$g->client_id}";

                $response = Http::withHeaders($this->http_headers)
                    ->withBody(json_encode($to_post), 'application/json')
                    ->post("{$this->url}/loans/{$loan->loan_id}/guarantors");
                if($response->ok())
                {
                    $api_response = json_decode($response, true);
                    $affected = DB::table('loan_guarantors')
                        ->where('id', $g->id)
                        ->update(['remote_id' => $api_response['resourceId'], 'sync' => 1]);
                    if($affected == 1)
                    {
                        echo "The guarantor of the loan application was updated in the database";
                    } else {
                        echo "The guarantor of the loan application was created in Ebanqr but not updated in the database";
                    }
                }
            }
        }
        echo "\n{$this->signature} is finished running \n";
    }
}
