<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\APITransaction;
use Illuminate\Support\Facades\Http;
use App\Models\CachedUser;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;


class LoanPostToEbanqr extends Command
{
    protected $signature = 'LoanPostToEbanqr:cron';

    protected $description = 'Post loan collateral application to Ebanqr';

    private $http_headers;

    private $url;

    public function __construct()
    {
        parent::__construct();

        $this->http_headers = [
            'Authorization' => env('EB_AUTHORIZATION'),
            'Accept' => env('EB_ACCEPT'),
            'api_key' => env('EB_API_KEY'),
            'eBanqr-Platform-TenantID' => env('EB_TENANTID'),
        ];
        $this->url = env('EB_URL');
    }

    public function handle()
    {
        echo "{$this->signature} is running \n";

        $loans = DB::table('loans')
                    ->where('sync', '=', 0)
                    ->get();

        foreach ($loans as $l) {
            #loan object to be posted to ebanqr
            $user = DB::table('users')
                        ->where('id' , '=', $l->user_id)
                        ->get()
                        ->first();

            $response = Http::withHeaders($this->http_headers)
                ->get("{$this->url}/savingsaccounts/{$user->account}");

            if($response->ok())
            {
                $data = json_decode($response);
            }


            $response2 = Http::withHeaders($this->http_headers)
                ->get("{$this->url}/loanproducts/{$l->loan_product}");

            if($response2->ok())
            {
                $data2 = json_decode($response2);
            }


            $loan_term_frequency_type = null;

            if($l->interval == 'Days')
            {
                $loan_term_frequency_type = 0;
            } elseif ($l->interval == 'Weeks')
            {
                $loan_term_frequency_type = 1;
            } elseif ($l->interval == 'Months')
            {
                $loan_term_frequency_type = 2;
            } elseif ($l->interval == 'Years')
            {
                $loan_term_frequency_type = 3;
            }

            $to_post = [
                "dateFormat" => "dd MMMM yyyy",
                "locale" => "en",
                "clientId" => $data->clientId,
                "productId" => intval($l->loan_product),
                "principal" => $l->loan_amount,
                "loanTermFrequency" => $data2->numberOfRepayments,
                "loanTermFrequencyType" => $loan_term_frequency_type,
                "loanType" => 'individual',
                "numberOfRepayments" => $data2->numberOfRepayments,
                "repaymentEvery" => $data2->repaymentEvery,
                "repaymentFrequencyType" => $data2->repaymentFrequencyType->id,
                "interestRatePerPeriod" => $data2->interestRatePerPeriod,
                "amortizationType" => $data2->amortizationType->id,
                "interestType" => $data2->interestType->id,
                "interestCalculationPeriodType" => $data2->interestCalculationPeriodType->id,
                "transactionProcessingStrategyId" => $data2->transactionProcessingStrategyId,
                "expectedDisbursementDate" =>  Carbon::parse($l->created_at)->format('d M Y'),
                "submittedOnDate" => Carbon::parse($l->created_at)->format('d M Y'),
            ];



            echo "Synching loan amount {$l->loan_amount} \n";

            $response = Http::withHeaders($this->http_headers)
                ->withBody(json_encode($to_post), 'application/json')
                ->post("{$this->url}/loans");


            if($response->ok()) {
                $api_response = json_decode($response, true);
                $affected = DB::table('loans')
                    ->where('id', $l->id)
                    ->update(['loan_id' => $api_response['loanId'], 'sync' => 1]);
                if($affected == 1)
                {
                    echo "The loan application was updated in the database";
                } else {
                    echo "The loan was created in Ebanqr but not updated the database ";
                }
            }

        }


        echo "\n{$this->signature} is finished running \n";
    }
}
