<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\APITransaction;
use Illuminate\Support\Facades\Http;
use App\Models\CachedUser;

class PostToEbanqr extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'PostToEbanqr:cron';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Post api transactions to Ebanqr';

    private $http_headers;
    private $url;

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();

        $this->http_headers = [
            'Authorization' => env('EB_AUTHORIZATION'),
            'Accept' => env('EB_ACCEPT'),
            'api_key' => env('EB_API_KEY'),
            'eBanqr-Platform-TenantID' => env('EB_TENANTID'),
        ];
        $this->url = env('EB_URL');
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        echo "{$this->signature} is running \n";

        #get pending transactions
        $transactions = APITransaction::where('sync', 'EB_SYNC_PENDING')->get();

        #loop through pending transactions
        foreach ($transactions as $t) {

            #post object/array to be posted to ebanqr
            $to_post = [
                "locale" => $t->locale,
                "dateFormat" => $t->dateFormat,
                "transactionDate" => $t->transactionDate,
                "transactionAmount" => $t->transactionAmount,
                "paymentTypeId" => $t->paymentTypeId,
                "accountNumber" => $t->accountNumber,
                "receiptNumber" => $t->receiptNumber,
            ];

            echo "Synching records for {$t->accountNumber} for amount {$t->transactionAmount}\n";

            #make post request to ebanqr to post the transaction
            $response = Http::withHeaders($this->http_headers)
                ->withBody(json_encode($to_post), 'application/json')
                ->post("{$this->url}/savingsaccounts/{$t->accountNumber}/transactions?command=deposit");

            #if the request gets posted
            if ($response->ok()) {

                #set sync to EB_SYNC_OK and comment to OK
                $t->sync = "EB_SYNC_OK";
                $t->comment = 'OK';

                #save transaction state
                $t->save();

                #get cached user 
                $cached_user = CachedUser::where('account_number', $t->accountNumber)->first();
                
                #if phone field is blank
                if($cached_user && $cached_user->phone == NULL){
                    echo "fetching phone from ebanqr\n";

                    #query ebanqr for it. 
                    $client_response = Http::withHeaders($this->http_headers)
                        ->get("{$this->url}/clients/{$cached_user->client_id}");

                    #if the response is okay
                    if($client_response->ok()){
                        #decode the response
                        $client_data = json_decode($client_response);
                        

                        #check if the phone number field exists
                        if(isset($client_data->mobileNo))
                        {
                            $cached_user->phone = str_replace(' ','',$client_data->mobileNo);
                            #save user phone details
                            $cached_user->save();
                        }
                        
                    }else{
                        #TODO: error loging
                        #something went wrong
                        echo "error in getting client detials from ebanqr\n";
                    }
                }

                #send sms if the person has a phone number
                if($cached_user->phone){
                    echo "sending sms\n";
                    #message template
                    $message = "Hi {client_name}, Your  A/C no:{account} recorded a deposit of UGX. {amount}. Date {date_time} Nzuri Trust";

                    #replace the message template with actual data
                    $message = str_replace(
                        ['{client_name}','{account}','{amount}','{date_time}'],
                        [$cached_user->name,$cached_user->account,number_format($t->transactionAmount),now()],
                        $message
                    );

                    #post fields
                    $sms_to_post =[
                        'user'=>'Nzuri',
                        'password'=>'trust1234',
                        'sender'=>'smswindow',
                        'message'=>$message,
                        'reciever'=>$cached_user->phone
                    ];

                    #make api call to send the sms message
                    $sms_response = Http::withHeaders(['Accept'=>'application/json'])
                            ->withBody(json_encode($sms_to_post), 'application/json')
                            ->post("http://smswindow.net/webapi.php");

                    var_export(json_decode($sms_response));
                }

            } else {
                #set sync flag to sync fail and save the response returned by 
                $t->sync = "EB_SYNC_FAILED";
                $t->comment = $response->json();
                
            }

            #save transaction state
            $t->save();
        }
        echo "\n{$this->signature} is finished running \n";
        return 0;
    }
}
