<?php

namespace App\Http\Controllers;

use App\Models\ApprovalRecord;
use App\Models\Approver;
use App\Models\ChargeProduct;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Loan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;


class LoanApplicationController extends Controller
{
    //
    public function __construct()
    {
        $this->http_headers = [
            'Authorization' => env('EB_AUTHORIZATION'),
            'Accept' => env('EB_ACCEPT'),
            'api_key' => env('EB_API_KEY'),
            'eBanqr-Platform-TenantID' => env('EB_TENANTID'),
        ];

        $this->url = env('EB_URL');

        $this->middleware('auth');
    }

    public function index()
    {
        $client_response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loanproducts");
        if($client_response->ok())
        {
            $data = json_decode($client_response);
        }
        $user = Auth::user();
        $module = "LoanApplication";
        $loans = $user->loans()->orderBy('loans.created_at', 'desc')->get();

        foreach ($loans as $loan)
        {
            if(!is_null($loan->loan_id))
            {
                $loan_response = Http::withHeaders($this->http_headers)
                    ->get("{$this->url}/loans/{$loan->loan_id}");
                if($loan_response->ok())
                {
                    $data2 = json_decode($loan_response);
                }
                DB::table('loans')
                    ->where('id', $loan->id)
                    ->update([
                        'status' => $data2->status->value,
                    ]);
            }

        }

        return view('loan_application.index',compact('user','module', 'loans', 'data'));
    }

    public function create()
    {
        $user = Auth::user();
        $module = "LoanApplication";

        $user_model = new User();

        return view('loan_application.create_loan', compact('user', 'module', 'user_model'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'loan_amount' => 'required',
            'loan_purpose' => 'required|in:working_capital_finance,project_finance,school_fees_study_loan,home_loans,asset_acquisition,agriculture,other',
            'other_loan_type' => 'required_if:loan_purpose,other',
            'security_offered' => 'required',
            'loan_product' => 'required|in:emergency_day_loan,express_loan,business_loan,group_loans,salary_loans,personal_loans',
            'interval' => 'required',
        ]);

        $loan_amount = $request['loan_amount'];
        $max_approval_level = null;



        if ($loan_amount >= 5000 && $loan_amount <= 1000000) {
            $max_approval_level = 1;
        } elseif ($loan_amount > 1000000 && $loan_amount < 2000000) {
            $max_approval_level = 2;
        } elseif ($loan_amount >= 2000000) {
            $max_approval_level = 3;
        }


        $loan_model = Loan::create([
            'user_id' => $user['id'],
            'loan_amount' => $loan_amount,
            'loan_purpose' => $request['loan_purpose'] === 'other' ? $request['other_loan_type'] : $request['loan_purpose'],
            'security_offered' => $request['security_offered'],
            'loan_product' => $request['loan_product'],
            'interval' => $request['interval'],
            'max_approval_level' => $max_approval_level
        ]);


        return redirect()->route('loan_application.index')->with('success', 'Loan successfully created!');
    }


    public function loan_approval()
    {
        $user = Auth::user();

        $loans = DB::table('approvers')
            ->join('loans', 'loans.approval_level', '=', 'approvers.approval_level')
            ->where('approvers.user_id', '=', $user['id'])
            ->where('loans.status', '=', 'PENDING')
            ->orderBy('loans.created_at', 'desc')
            ->get();

        $module = "LoanApproval";

        $allLoans = Loan::orderBy('created_at', 'desc')->get();

        return view('loan_application.loan_approval', compact('user', 'module', 'loans', 'allLoans'));
    }

    public function approve($loanId, Request $request)
    {
        $loan = Loan::findOrFail($loanId);
        if (!$loan) {
            return redirect()->route('loan_application.loan_approval')->with('error', 'Loan not found.');
        }
        $user = Auth::user();
        $user_level = Approver::find($user->id);
        $currentLevel = $loan->approval_level;
        $maxLevel = $loan->max_approval_level;
        $request->validate([
            'amount' => 'required',
            'comment' => 'required|string',
        ]);

        if ($currentLevel < $maxLevel) {
            $loan->approval_level++;
            $loan->save();
            $approval_record = ApprovalRecord::create([
                'loan_id' => $loanId,
                'user_id' => $user->id,
                'level' => $user_level->user->approval_level,
                'adjusted_amount' => $request['amount'],
                'comments' => $request['comment'],
                'approval_level' => $user_level->approval_level

            ]);
            return redirect()->route('loan_application.loan_approval')->with('success', 'Loan approved and moved to the next level.');
        } else {
            $loan->status = 'approved';
            $loan->save();
            $approval_record = ApprovalRecord::create([
                'loan_id' => $loanId,
                'user_id' => $user->id,
                'adjusted_amount' => $request['amount'],
                'comments' => $request['comment'],
                'approval_level' => $user_level->approval_level

            ]);
            return redirect()->route('loan_application.loan_approval')->with('success', 'Loan approved successfully.');
        }
    }

    public function reject(Loan $loan, Request $request)
    {
        // Check if the loan exists
        if (!$loan) {
            return redirect()->route('loan_application.loan_approval')->with('error', 'Loan not found.');
        }

        // Validate the request (add more validation rules as needed)
        $request->validate([
            'comment' => 'required|string',
        ]);

        // Update the loan status to "rejected" and store the comment
        $loan->update([
            'status' => 'rejected',
            'comment' => $request->input('comment'),
        ]);

        // Redirect back to the index page or wherever you want
        return redirect()->route('loan_application.loan_approval')->with('warning', 'Loan rejected successfully.');
    }

    public function product(Request $request)
    {
        $user = Auth::user();

        $client_response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loanproducts/{$request['loan_product']}");
        if($client_response->ok())
        {
            $data = json_decode($client_response);
        }

        $loan_interval = null;

        $loan = Loan::create([
            'user_id' => $user['id'],
            'loan_product' => $request['loan_product'],
            'loan_product_name' => $data->name,
            'interval' => $data->repaymentFrequencyType->value,
        ]);

        $loan_id = $loan->id;

        return redirect()->route('terms.index', compact('loan_id'))->with('success', 'Product successfully created');
    }

    public function terms($loan_id)
    {
        $loan = DB::table('loans')->where('id', $loan_id)->first();
        $response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loanproducts/{$loan->loan_product}");
        if($response->ok())
        {
            $loan_product = json_decode($response);
        }
        $user = Auth::user();
        $module = 'LoanApplication';
        return view('loan_application.wizard.terms', compact('user','module', 'loan_id', 'loan', 'loan_product'));
    }

    public function post_terms(Request $request)
    {
        $user = Auth::user();
        $loan_id = $request['loan_id'];
        $loan = DB::table('loans')->where('id', $request['loan_id'])->get()->first();
        $response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loanproducts/{$loan->loan_product}");
        if($response->ok())
        {
            $loan_product = json_decode($response);
        }
        $request->validate([
            'loan_principle' => 'required|numeric|min:' . $loan_product->minPrincipal . '|max:' . $loan_product->maxPrincipal,
            'loan_term' => 'required',
//            'loan_repayment' => 'required',
//            'loan_repayment_frequency' => 'required'
        ]);
        DB::table('loans')
            ->where('id', $request['loan_id'])
            ->update([
                'loan_amount' => $request['loan_principle'],
                'loan_term' => $request->input('loan_term'),
//                'number_of_repayments' => $request->input('loan_repayment'),
//                'repayment_frequency' => $request->input('loan_repayment_frequency')
            ]);

        if(is_null($loan->loan_id))
        {
            $response = Http::withHeaders($this->http_headers)
                ->get("{$this->url}/savingsaccounts/{$user->account}");
            if($response->ok())
            {
                $data = json_decode($response);
            }

            $response2 = Http::withHeaders($this->http_headers)
                ->get("{$this->url}/loanproducts/{$loan->loan_product}");

            if($response2->ok())
            {
                $data2 = json_decode($response2);
            }

            $loan_term_frequency_type = null;

            if($loan->interval == 'Days')
            {
                $loan_term_frequency_type = 0;
            } elseif ($loan->interval == 'Weeks')
            {
                $loan_term_frequency_type = 1;
            } elseif ($loan->interval == 'Months')
            {
                $loan_term_frequency_type = 2;
            } elseif ($loan->interval == 'Years')
            {
                $loan_term_frequency_type = 3;
            }
            $to_post = [
                "dateFormat" => "dd MMMM yyyy",
                "locale" => "en",
                "clientId" => $data->clientId,
                "productId" => intval($loan->loan_product),
                "principal" => $request['loan_principle'],
                "loanTermFrequency" => $request['loan_term'],
                "loanTermFrequencyType" => $loan_term_frequency_type,
                "loanType" => 'individual',
                "numberOfRepayments" => $request['loan_term'],
                "repaymentEvery" => $data2->repaymentEvery,
                "repaymentFrequencyType" => $data2->repaymentFrequencyType->id,
                "interestRatePerPeriod" => $data2->interestRatePerPeriod,
                "amortizationType" => $data2->amortizationType->id,
                "interestType" => $data2->interestType->id,
                "interestCalculationPeriodType" => $data2->interestCalculationPeriodType->id,
                "transactionProcessingStrategyId" => $data2->transactionProcessingStrategyId,
                "expectedDisbursementDate" =>  Carbon::parse($loan->created_at)->format('d M Y'),
                "submittedOnDate" => Carbon::parse($loan->created_at)->format('d M Y'),
                "linkAccountId" => $data->id
            ];
            $response3 = Http::withHeaders($this->http_headers)
                ->withBody(json_encode($to_post), 'application/json')
                ->post("{$this->url}/loans");
            if($response->ok()) {
                $api_response = json_decode($response3, true);
                $affected = DB::table('loans')
                    ->where('id', $loan->id)
                    ->update(['loan_id' => $api_response['loanId'], 'sync' => 1]);
                if($affected == 1)
                {
                    return redirect()->route('settings.index', compact('loan_id'))->with('success', 'success');
                } else {
                    echo "The loan was created in Ebanqr but not updated the database ";
                }
            } else {
                echo "there was an error syncing data to Ebanqr";
            }
        }
        return redirect()->route('settings.index', compact('loan_id'))->with('success', 'success');
    }

    public function settings($loan_id)
    {
        $loan = DB::table('loans')->where('id', $loan_id)->first();
        $user = Auth::user();
        $module = 'LoanApplication';
        $first_repayment_on = '';
        if($loan->interval === 'Months')
        {
            $first_repayment_on = Carbon::parse($loan->created_at)->copy()->addMonth()->toDateString();
        } elseif ($loan->interval === 'Weeks') {
            $first_repayment_on = Carbon::parse($loan->created_at)->copy()->addWeek()->toDateString();
        }

        return view('loan_application.wizard.settings', compact('user', 'module', 'loan_id', 'loan', 'first_repayment_on'));
    }

    public function post_settings(Request $request)
    {
        $loan_id = $request['loan_id'];
        DB::table('loans')
            ->where('id', $request['loan_id'])
            ->update([
                'loan_purpose' => $request['loan_purpose'],
                'loan_classification' => $request['loan_classification'],
                'standing_instruction' => $request['standing_instruction'],
                'is_top_up' => $request['top_up'],
                'linked_Account' => $request['loan_savings_account'],
                'first_repayment_date' => $request['first_repayment_date']
            ]);
        return redirect()->route('charges.index', compact('loan_id'))->with('success', 'success');
    }

    public function charges($loan_id)
    {
        $totalLoanCharges = null;

        $loan = DB::table('loans')->where('id', $loan_id)->first();

        if(is_null($loan->charges_sync))
        {
            //get me the charges of the specific loan as set by the admin
            $charges = DB::select("SELECT * from charge_product WHERE product_id = ? ORDER BY created_at DESC", [$loan->loan_product]);

            $syncArray = [
                'charges' => count($charges),
                'sync' => []
            ];
            foreach ($charges as $charge)
            {
                $dueDate = null;

                //calculate the duedate I am assuming the duedate to be the first repayment date
                if($loan->interval === 'Months')
                {
                    $dueDate = Carbon::parse($loan->created_at)->copy()->addMonth()->toDateString();
                } elseif ($loan->interval === 'Weeks') {
                    $dueDate = Carbon::parse($loan->created_at)->copy()->addWeek()->toDateString();
                }

                //I make an api request to get the amount of the charge
                $charge_response = Http::withHeaders($this->http_headers)
                    ->get("{$this->url}/charges/{$charge->charge_id}");

                if($charge_response->ok())
                {
                    $data = json_decode($charge_response);
                }

                $to_post = [
                    "chargeId" => $charge->charge_id,
                    "amount" => $data->amount,
                    "dueDate" => Carbon::parse($dueDate)->format('d M Y'),
                    "dateFormat" => "dd MMMM yyyy",
                    "locale" => "en",
                ];
                $post_response = Http::withHeaders($this->http_headers)
                    ->withBody(json_encode($to_post), 'application/json')
                    ->post("{$this->url}/loans/{$loan->loan_id}/charges");
                if($post_response->ok())
                {
                    $syncArray['sync'][$charge->charge_id] = "OK";
                } else {
                    $syncArray['sync'][$charge->charge_id] = "ERROR";
                }
            }

            //I update the loan with the sync object
            DB::table('loans')
                ->where('id', $loan_id)
                ->update(['charges_sync' => json_encode($syncArray)]);
        }

        $response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loans/{$loan->loan_id}");
        if($response->ok())
        {
            $api_response = json_decode($response, true);
            $totalLoanCharges = $api_response;
        }

        $user = Auth::user();
        $module = 'LoanApplication';


        return view ('loan_application.wizard.charges', compact('user', 'module', 'totalLoanCharges', 'loan_id', 'loan'));
    }

    public function collateral($loan_id)
    {
        $loan = DB::table('loans')->where('id', $loan_id)->first();
        $client_response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loans/{$loan->loan_id}/collaterals/template");

        if($client_response->ok())
        {
            $data = json_decode($client_response);
        }

        $user = Auth::user();
        $collateral = DB::table('loan_collateral')
            ->where('user_id', $user['id'])
            ->where('loan_id', $loan_id)
            ->get();
        $module = 'LoanApplication';
        return view('loan_application.wizard.collateral', compact('user', 'module', 'loan_id', 'collateral', 'loan', 'data'));
    }

    public function post_collateral(Request $request)
    {
        $user = Auth::user();
        $loan_id = $request['loan_id'];


        DB::table('loan_collateral')->insert([
            'user_id' => $user['id'],
            'loan_id' => $loan_id,
            'collateral_name' => $request['collateral_name'],
            'collateral_type' => $request['collateral_type'],
            'value' => $request['collateral_value'],
            'description' => $request['collateral_description']
        ]);

        return redirect()->route('collateral.index', compact('loan_id'))->with('success', 'success');
    }

    public function guarantor($loan_id)
    {
        $internal_guarantors = null;
        $external_guarantors = null;
        $loan = DB::table('loans')->where('id', $loan_id)->first();
        $client_response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loans/{$loan->loan_id}/guarantors/template");

        if($client_response->ok())
        {
            $data = json_decode($client_response);
        }
        $loan = DB::table('loans')->where('id', $loan_id)->first();
        $user = Auth::user();
        $loan_guarantors = DB::table('loan_guarantors')
            ->where('user_id', $user['id'])
            ->where('loan_id', $loan_id)
            ->get();
        foreach ($loan_guarantors as $guarantor)
        {
            if(is_null($guarantor->client_id))
            {
                $internal_guarantors[] = $guarantor;
            } else {

                $response = Http::withHeaders($this->http_headers)
                    ->get("{$this->url}/clients/{$guarantor->client_id}");
                if($response->ok())
                {
                    $external_guarantors[] = json_decode($response);
                }

            }
        }

        $module = 'LoanApplication';
        return view('loan_application.wizard.guarantor', compact('user', 'module', 'loan_id', 'internal_guarantors','external_guarantors', 'loan', 'data'));
    }

    public function search_customer($searchTerm)
    {
        $client_response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/search?query={$searchTerm}");
        if($client_response->ok())
        {
            $data = json_decode($client_response);
        }

        return $data;
    }

    public function post_guarantor(Request $request)
    {
        $user = Auth::user();
        $loan_id = $request['loan_id'];
        if($request['clientType'] === 'NO')
        {
            DB::table('loan_guarantors')->insert([
                'user_id' => $user['id'],
                'loan_id' => $loan_id,
                'relationship' => $request['guarantor_relationship'],
                'first_name' => $request['guarantor_first_name'],
                'last_name' => $request['guarantor_last_name'],
                'date_of_birth' => $request['guarantor_dob'],
                'address' => $request['guarantor_address'],
                'city' => $request['guarantor_city'],
                'phone_number' => $request['guarantor_tel'],
                'guarantor_type_id' => 3,
                'client_relationship_type_id' => $request['guarantor_relationship'],
            ]);
        } elseif ($request['clientType'] === 'YES')
        {
            DB::table('loan_guarantors')->insert([
                'user_id' => $user['id'],
                'loan_id' => $loan_id,
                'guarantor_type_id' => $request['guarantor_type'],
                'relationship' => $request['guarantor_relationship_existing'],
                'client_relationship_type_id' => $request['guarantor_relationship_existing'],
                'client_id' => $request['guarantor_client_id']
            ]);
        }
        return redirect()->route('guarantor.index', compact('loan_id'))->with('success', 'success');
    }

    public function repayment_schedule($loan_id)
    {
        $user = Auth::user();

        $loan = DB::table('loans')->where('id', $loan_id)->first();

        $response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loans/{$loan->loan_id}?associations=repaymentSchedule");

        if($response->ok())
        {
            $data = json_decode($response, true);
        }

        $loan_schedule_response = $data['repaymentSchedule'];
        $loan_schedule = $loan_schedule_response['periods'];
        $currency = $loan_schedule_response['currency']['code'];
        $total_principle = $loan_schedule_response['totalPrincipalExpected'];
        $expected_amount = $loan_schedule_response['totalRepaymentExpected'];
        $total_interests = $loan_schedule_response['totalInterestCharged'] + $loan_schedule_response['totalFeeChargesCharged'];
        $module = 'LoanApplication';
        return view('loan_application.wizard.schedule', compact('user', 'module', 'loan_id', 'loan', 'loan_schedule', 'currency', 'total_principle','expected_amount','total_interests'));
    }

    public function loan_preview($loan_id)
    {
        $user = Auth::user();
        $loan = DB::table('loans')->where('id', $loan_id)->first();

        $collateral = DB::table('loan_collateral')
            ->where('user_id', $user['id'])
            ->where('loan_id', $loan_id)
            ->get();

        $loan_guarantors = DB::table('loan_guarantors')
            ->where('user_id', $user['id'])
            ->where('loan_id', $loan_id)
            ->get();
        $user = Auth::user();
        $module = 'LoanApplication';
        return view('loan_application.wizard.preview', compact('user', 'module', 'loan_id','loan','collateral','loan_guarantors'));
    }

    public function loan_charges_admin()
    {
        $charges_to_view = [];
        $charges = DB::table('charge_product')->get();
        if($charges->isNotEmpty())
        {
            foreach ($charges as $charge)
            {
                $charge_response = Http::withHeaders($this->http_headers)
                    ->get("{$this->url}/charges/{$charge->charge_id}");

                if($charge_response->ok())
                {
                    $data = json_decode($charge_response);
                }

                $product_response = Http::withHeaders($this->http_headers)
                    ->get("{$this->url}/loanproducts/{$charge->product_id}");

                if($product_response->ok())
                {
                    $data2 = json_decode($product_response);
                }

                $charges_to_view[]= [
                    'charge_name' => $data->name,
                    'product_name' => $data2->name,
                ];
            }
        }

        $user = Auth::user();
        $module = 'LoanCharges';
        return view('loan_application.charges_admin', compact('user', 'module', 'charges_to_view'));
    }

    public function loan_charges_create()
    {
        $user = Auth::user();
        $this->authorize('add-loan-charge', $user);
        $module = "LoanCharges";

        $products_response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loanproducts");

        if($products_response->ok())
        {
            $data = json_decode($products_response);
        }

        $products_template_response = Http::withHeaders($this->http_headers)
            ->get("{$this->url}/loanproducts/template");

        if($products_template_response->ok())
        {
            $products_template = json_decode($products_template_response);
        }




        return view('loan_application.add_loan_charge', compact('user','module', 'data', 'products_template'));
    }

    public function loan_charges_store(Request $request)
    {
        $charge_product_model = ChargeProduct::create([
            'charge_id' => $request['loan_charge'],
            'product_id' => $request['loan_product'],
        ]);

        return redirect()->route('loan_application.loan_charges_admin')->with('success', 'Loan Charge Added Successfully');
    }
}
